#########################################################
#
# Function to compute the Asymptotic variance of VN MCV estimate
#
#########################################################
library(mvtnorm)
#library(rrcov)
library(normalp)
library(MNM)

# Files needed to run this script
# - consistMCD
# - consistS
# - fastS_normality
# - fastS_consistency

# ------------- Main function

ASVcvVN<- function(g ,p , dist="norm",df=0, estim ="class",bdp=0.25){
  # INPUT:
  # g : estimated or theoretical MCV 
  # p: dimension
  # dist: distribution "norm", "stud" or "powerexp"
  # df: degree of freedom for the stud or porwerexp distribution
  #       NB: for the power exp, df=1 corresponds to the normal distribution
  # estim : location and scatter estimators 
  #     'class' : sample estimate
  #     'IQRmed' : (p=1) IQR and median
  #     'MADmed' : (p=1) MAD and median
  #     'MCD'   : MCD estimate (raw)
  #     'RMCD'  : one-step reweighted MCD estimate
  #     'S'     : S estimate with Tukey's biweight function
  # bdp: BDP 
  
  # OUTPUT 
  # ASV : corresponding asymptotic variance
  
  alpha=1-bdp
  delta=0.025
  Nasv=50000
  # -------------Univariate
  if(p==1){
    if(estim=="class"){
      if(dist=="norm"){
        kurtosis<- 0
      }else if(dist=="stud"){ # defined such that COV(x)=sigma^2
        kurtosis<-2/(df-4)
      }else if(dist=="powerexp"){ # defined such that COV(x)=sigma^2
        kurtosis<- p/(p+2)* gamma(p/(2*df))*gamma((p+4)/(2*df))/gamma((p+2)/(2*df))^2 -1
      }   
      C1<- 1
      C2<- 0.25*(2 + 3* kurtosis)
    }else if(estim=="IQRmed"){ 
      if(dist=="norm"){
        consist<-1/(2*qnorm(0.75))
        C1<- 1/(4*dnorm(0)^2)
        C2<- consist^2/(4*dnorm(qnorm(0.75))^2)   
      }else if(dist=="stud"){
        consist<-1/(2*sqrt((df-2)/df)*qt(0.75,df))
        C1<- 1/(4*df/(df-2)*dt(0,df)^2)
        C2<- consist^2/(4*(df/(df-2))*dt(qt(0.75,df),df)^2)
      }else if(dist=="powerexp"){
        a<-(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
        consist<- 1/(2*qnormp(0.75,mu=0,sigmap=1/(2*a*df)^(1/(2*df)), p=2*df))
        C1<- 1/(4*dnormp(0,mu=0,sigmap=1/(2*a*df)^(1/(2*df)),p=2*df)^2)
        C2<- consist^2/(4*dnormp(qnormp(0.75,mu=0,sigmap=1/(2*a*df)^(1/(2*df)), p=2*df),mu=0,sigma=1/(2*a*df)^(1/(2*df)), p=2*df)^2)
      }
    }else if(estim=="MADmed"){
      if(dist=="norm"){
        consist<- 1/ qnorm(0.75)
        C1<- 1/(4*dnorm(0)^2)
        C2<- consist^2/(16 *dnorm(qnorm(0.75))^2 ) 
      }else if(dist=="stud"){
        consist<- 1/(sqrt((df-2)/df)* qt(0.75,df))
        C1<- 1/(4*df/(df-2)*dt(0,df)^2)
        C2<- consist^2/(16 *(df/(df-2))*dt(qt(0.75,df),df)^2 )
      }else if(dist=="powerexp"){
        a<-(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
        consist<- 1/qnormp(0.75,mu=0,sigmap=1/(2*a*df)^(1/(2*df)), p=2*df)
        C1<- 1/(4*dnormp(0,mu=0,sigmap=1/(2*a*df)^(1/(2*df)),p=2*df)^2)
        C2<- consist^2/(16*dnormp(qnormp(0.75,mu=0,sigmap=1/(2*a*df)^(1/(2*df)), p=2*df),mu=0,sigma=1/(2*a*df)^(1/(2*df)), p=2*df)^2)
      }
    }
    else stop("This is not a valid estimator")
    ASV<- C1 * g^4+ C2* g^2
  }# -----------------------Multivariate
  else if(p>1) {
    if(dist=="norm") # OK
    {
      if(estim=="class"){
        tau<-1
        sigma1<-1
        sigma2<-0 
        Cte<-2*sigma1+sigma2
      }else if(estim=="MCD"){   
        # alpha corresponds the mass of observations used in the MCD computation 
        # with constants from Cator and Lopuhaa (2010)
        qalpha<- qchisq(alpha,df=p)
        r<-sqrt(qalpha)
        sq_calpha<- sqrt(pchisq(qalpha,df=p+2)/alpha)
        calpha<-sq_calpha^2
        nu0<- 2^(3/2)*gamma((p+1)/2)/gamma(p/2)*sq_calpha*dchisq(qalpha,df=p+1)
        k3<- -sq_calpha^3*p*(p+2)/(r^3*nu0-alpha*p*(p+2)*sq_calpha^3)
        tau<- p^2*alpha*sq_calpha^4/(p*alpha*sq_calpha-r*nu0)^2
        sigma1<- k3^2 * pchisq(qalpha,df=p+4)
        sigma2<- -2*sigma1/p + (p+2)/(p*alpha^2)*pchisq(qalpha,df=p+4) - 
          1/(alpha*p^2)*(alpha*r^4-2*p*alpha*r^2*sq_calpha^2+p^2*alpha*sq_calpha^4+2*p*r^2*sq_calpha^2-r^4)
        sigma1<-1/sq_calpha^4*sigma1
        sigma2<-1/sq_calpha^4*sigma2
        Cte<-2*sigma1+sigma2
        
      }else if(estim=="RMCD"){
        qdelta<-qchisq(1-delta,p)
        d1<-  1-delta
        d2<-  d1 + 2*(-pchisq(qdelta,p+2)/2)
        d3<-  pchisq(qdelta,p+2)
        d4<-  d3 - pchisq(qdelta,p+4)  
        # Computation of the tau and 2*sigma_1 + sigma_2 constants numerically using the IF
        # since tau = mean(IF(ech, T, F)^2)[1,1] where F is the stadard distribution ( mean=0, cov= I)
        # and 2*sigma_1 + sigma_2 = mean(IF(ech,C,F)[1,1]^2) where F is the standard distribution ( mean=0, cov= I) 
        ech<-rmvnorm(n=Nasv,mean=rep(0,p))  # standard normal distribution
        # Use the IF function version where the constants d1, d2,d3,d4 and qdelta are provided
        # so that these constants , that does not depend on the sample but only on the distribution, dimension and BDP, are 
        # not recomputed for each observation
        tau<-mean(unlist(lapply(myIFrewMCDLoc_matbis(ech,mu=rep(0,p),sigma=diag(p),mod="norm",alpha=bdp,qdelta=qdelta,d1=d1,d2=d2),function(x)(x%*%t(x))[1,1])))
        Cte<-mean(unlist(lapply(myIFrewMCDCov_mat(ech,mu=rep(0,p),sigma=diag(p),mod="norm",alpha=bdp,delta=delta),function(x)(x[1,1])^2)))
        
      } else if(estim=="S"){
        c0 <- csolve.bw.asymp(p,bdp)
        b0<- erho.bw(p,c0)
        
        # tau constant
        tmp_alpha<- 1/p*chi.int(p,2,c0) + 6/(p*c0^4)*chi.int(p,6,c0) +1/(p*c0^8)*chi.int(p,10,c0) -
          4/(p*c0^2)*chi.int(p,4,c0) -4/(p*c0^6)*chi.int(p,8,c0)
        tmp_beta<-  chi.int(p,0,c0) + 1/p* (- 2*(p+2)/c0^2*chi.int(p,2,c0) + (p+4)/c0^4*chi.int(p,4,c0) )
        tau<-tmp_alpha/tmp_beta^2
        
        # sigma1 constant
        g1<- 1/(p+2) *( (p+2)*chi.int(p,2,c0) - 2/c0^2*(p+4)*chi.int(p,4,c0) + 1/c0^4*(p+6)*chi.int(p,6,c0))
        sigma1<-p/(p+2)*1/g1^2 *(chi.int(p,4,c0)-4/c0^2*chi.int(p,6,c0) +6/c0^4*chi.int(p,8,c0) -
                                   4/c0^6*chi.int(p,10,c0) +1/c0^8*chi.int(p,12,c0))
        # sigma2 constant
        tmp_num<- chi.int(p,4,c0)/4 - 1/(2*c0^2)*chi.int(p,6,c0) + 5/(12*c0^4)*chi.int(p,8,c0) -
          1/(6*c0^6)*chi.int(p,10,c0) +1/(36*c0^8)*chi.int(p,12,c0) + c0^4/36* chi.int2(p,0,c0)-b0^2
        g3<- chi.int(p,2,c0) - 2/c0^2*chi.int(p,4,c0) + 1/c0^4*chi.int(p,6,c0)
        sigma2<- -2/p*sigma1 +4*tmp_num/g3^2
        
        Cte<-2*sigma1+sigma2
      }else{stop("ASV for this estimator is not available")}  
      
    }else if(dist=="stud"){ 
      # Defined such that COV(X) = Sigma !
      if(estim=="class"){
        kurtosis<-2/(df-4)
        tau<-1
        sigma1<-1+kurtosis
        sigma2<-kurtosis
        Cte<-2*sigma1+sigma2
        
      }else if(estim=="MCD"){
        alpha<-(1-bdp)
        K<- (df-2)^(-p/2)*gamma((p+df)/2)/(pi^(p*0.5)*gamma(df/2))
        const<- (1-bdp)*gamma(0.5*p)/(2*pi^(p*0.5))*1/K
        qalpha<-uniroot(stud_int2,interval=c(0,50),p=p,df=df,const=const)$root
        r<-sqrt(qalpha)
        calpha<-1/const *1/p*stud_funct(qalpha,e=2,p=p,df=df) # consistency factor = 1/calpha
        nu0=(1-bdp)*sqrt(calpha)/const *stud_integrand(r,e=0,p=p,df=df)
        condMom4=(1-bdp)/const *stud_funct(qalpha,e=4,df=df,p=p)
        
        rho<-r/sqrt(calpha)
        beta2<-2*rho^3*nu0/(sqrt(calpha)*p*(p+2))-2*(1-bdp)/sqrt(calpha)
        #k3<- -calpha^(3/2)*p*(p+2)/(r^3*nu0-alpha*p*(p+2)*calpha^(3/2))
        k3=-2/(sqrt(calpha)*beta2)
        
        tau<- p^2*(1-bdp)*calpha^2/(p*(1-bdp)*sqrt(calpha)-r*nu0)^2
        sigma1<- k3^2 *condMom4/(p*(p+2))
        sigma2<- -2*sigma1/p + 1/(p^2*alpha^2)*condMom4 - 1/(alpha*p^2)*(alpha*r^4-2*p*alpha*r^2*calpha+p^2*alpha*calpha^2+2*p*r^2*calpha-r^4)
        sigma1<-1/calpha^2*sigma1
        sigma2<-1/calpha^2*sigma2
        Cte<-2*sigma1+sigma2
        
      }else if(estim=="RMCD"){
        K<- (df-2)^(-p/2)*gamma((p+df)/2)/(pi^(p*0.5)*gamma(df/2))
        const<- (1-delta)*gamma(0.5*p)/(2*pi^(p*0.5))*1/K
        qdelta<-uniroot(stud_int2,interval=c(0,30),p=p,df=df,const=const)$root
        d1<- 1/const*(1-delta)*stud_funct(qdelta,e=0,p=p,df=df)
        d2<- d1+ 2*1/const*(1-delta)/p*(-0.5*(df+p))/(df-2)*stud_funct2(qdelta,e=2,p=p,df=df)
        d3<- 1/const*(1-delta)/p*stud_funct(qdelta,e=2,p=p,df=df)
        d4<- d3+ 1/const*(1-delta)*2/(p*(p+2))*(-0.5*(df+p))/(df-2) * stud_funct2(qdelta,e=4,p=p,df=df)
        # Computation of the tau and 2*sigma_1 + sigma_2 constants numerically using the IF
        # since tau = mean(IF(ech, T, F)^2)[1,1] where F is the stadard distribution ( mean=0, cov= I)
        # and 2*sigma_1 + sigma_2 = mean(IF(ech,C,F)[1,1]^2) where F is the standard distribution ( mean=0, cov= I)       
        ech<-rmvt(n=Nasv,sigma=(df-2)/df*diag(p),df=df) # sample form the standard exp distribution (mean=0, cov= I)
        # Use the IF function version where the constants d1, d2,d3,d4 and qdelta are provided
        # so that these constants , that does not depend on the sample but only on the distribution, dimension and BDP, are 
        # not recomputed for each observation
        tau<-mean(unlist(lapply(myIFrewMCDLoc_matbis(ech,mu=rep(0,p),sigma=diag(p),mod="stud",df=df,alpha=bdp,qdelta=qdelta,d1=d1,d2=d2),function(x)(x%*%t(x))[1,1])))
        Cte<-mean(unlist(lapply(myIFrewMCDCov_mat(ech,mu=rep(0,p),sigma=diag(p),mod="stud",df=df,alpha=bdp,delta=delta),function(x)(x[1,1])^2)))
        
      }else if(estim=="S") {
        c0<-uniroot(Sint2_stud,interval=c(1,50),p=p,df=df,bdp=bdp)$root
        b0<-bfct_stud(c0,p,df)
        
        # tau constant
        tmp_alpha<- 1/p*(Sfct_stud(c0,e=2,p,df) -4/c0^2*Sfct_stud(c0,e=4,p,df) + 6/c0^4*Sfct_stud(c0,e=6,p,df) -4/c0^6*Sfct_stud(c0,e=8,p,df) + 1/c0^8*Sfct_stud(c0,e=10,p,df))
        tmp_beta<-  -2/c0^2*(1 + 2/p)*Sfct_stud(c0,e=2,p,df) + 1/c0^4*(1 + 4/p)*Sfct_stud(c0,e=4,p,df)+Sfct_stud(c0,e=0,p,df)
        tau= tmp_alpha/tmp_beta^2
        
        # sigma1 constant
        g1<-Sfct_stud(c0,e=2,p,df)*(p+2)-2/c0^2*(p+4)*Sfct_stud(c0,e=4,p,df) +1/c0^4*(p+6)*Sfct_stud(c0,e=6,p,df)
        sigma1<- p*(p+2)*1/g1^2*(Sfct_stud(c0,e=4,p,df)-4/c0^2*Sfct_stud(c0,e=6,p,df)+6/c0^4*Sfct_stud(c0,e=8,p,df)-4/c0^6*Sfct_stud(c0,e=10,p,df) +1/c0^8*Sfct_stud(c0,e=12,p,df))
        
        # sigma2 constant 
        tmp_num<-0.25*Sfct_stud(c0,e=4,p,df) -1/(2*c0^2)*Sfct_stud(c0,e=6,p,df) +5/(12*c0^4)*Sfct_stud(c0,e=8,p,df)-1/(6*c0^6)*Sfct_stud(c0,e=10,p,df) +1/(36*c0^8)*Sfct_stud(c0,e=12,p,df)+c0^4/36*(1-Sfct_stud(c0,e=0,p,df))-b0^2
        g3<-Sfct_stud(c0,e=2,p,df)-2/c0^2*Sfct_stud(c0,e=4,p,df) +1/c0^4*Sfct_stud(c0,e=6,p,df)
        sigma2= -2/p*sigma1 +4*tmp_num/g3^2
        
        Cte<-2*sigma1+sigma2
      }else{stop("ASV for this estimator is not available")}
      
    }else if(dist=="powerexp") { 
      # Defined such that COV(X) = Sigma !
      if(estim=="class"){
        kurtosis<-p/(p+2)* gamma(p/(2*df))*gamma((p+4)/(2*df))/gamma((p+2)/(2*df))^2 -1
        tau<-1
        sigma1<-1+kurtosis
        sigma2<-kurtosis
        Cte<-2*sigma1+sigma2
        
      }else if(estim=="MCD") {
        const<-(1-bdp)*gamma(p/(2*df))/(2*df) *((p*gamma(p/(2*df)))/gamma((p+2)/(2*df)))^(0.5*p)
        a=(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
        qalpha<-uniroot(mvexp_int2,interval=c(1,50),p=p,df=df,const=const)$root
        r=sqrt(qalpha)
        calpha<-1/const*1/p*mvexp_funct(qalpha,e=2,p,df)
        
        nu0=(1-bdp)*sqrt(calpha)/const*mvexp_integrand(r,df=df,p=p,e=0)
        condMom4=(1-bdp)/const *mvexp_funct(qalpha,e=4,p=p,df=df)
        k3<- -calpha^(3/2)*p*(p+2)/(r^3*nu0-alpha*p*(p+2)*calpha^(3/2))
        
        tau<- p^2*alpha*calpha^2/(p*alpha*sqrt(calpha)-r*nu0)^2
        sigma1<- k3^2 *condMom4/(p*(p+2))
        sigma2<- -2*sigma1/p + 1/(p^2*alpha^2)*condMom4 - 
          1/(alpha*p^2)*(alpha*r^4-2*p*alpha*r^2*calpha+p^2*alpha*calpha^2+2*p*r^2*calpha-r^4)
        sigma1<-1/calpha^2*sigma1
        sigma2<-1/calpha^2*sigma2
        Cte<-2*sigma1+sigma2
        
      }else if(estim=="RMCD"){
        const<-(1-delta)*gamma(p/(2*df))/(2*df) *((p*gamma(p/(2*df)))/gamma((p+2)/(2*df)))^(0.5*p)
        a=(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
        qdelta<-uniroot(mvexp_int2,interval=c(1,30),p=p,df=df,const=const)$root
        d1<- 1/const*(1-delta)*mvexp_funct(qdelta, e=0,p=p,df=df)
        d2<-d1+ 2*1/const*(1-delta)/p *(-a*df)*mvexp_funct(qdelta,e=2*df,p=p,df=df)  
        d3<-1/const*(1-delta)/p*mvexp_funct(qdelta, e=2,p=p,df=df)
        d4<-d3+ 1/const*(1-delta)*2/(p*(p+2)) *(-a*df)*mvexp_funct(qdelta,e=2*df+2,p=p,df=df)    
        # Computation of the tau and 2*sigma_1 + sigma_2 constants numerically using the IF
        # since tau = mean(IF(ech, T, F)^2)[1,1] where F is the stadard distribution ( mean=0, cov= I)
        # and 2*sigma_1 + sigma_2 = mean(IF(ech,C,F)[1,1]^2) where F is the standard distribution ( mean=0, cov= I)       
        ech<- myrmvpowerexp(Nasv, Location=rep(0,p), Scatter= (2*a)^(-1/df)*diag(p), Beta=df) # standard exp distribution (mean=0, cov= I)
        # Use the IF function version where the constants d1, d2,d3,d4 and qdelta are provided
        # so that these constants , that does not depend on the sample but only on the distribution, dimension and BDP, are 
        # not recomputed for each observation
        tau<-mean(unlist(lapply(myIFrewMCDLoc_matbis(ech,mu=rep(0,p),sigma=diag(p),mod="powerexp",df=df,alpha=bdp,qdelta=qdelta,d1=d1,d2=d2),function(x)(x%*%t(x))[1,1])))
        Cte<-mean(unlist(lapply(myIFrewMCDCov_mat(ech,mu=rep(0,p),sigma=diag(p),mod="powerexp",df=df,alpha=bdp,delta=delta),function(x)(x[1,1])^2)))
        
      }else if(estim=="S"){
        c0<-uniroot(Sint2_mvexp,interval=c(1,50),p=p,df=df,bdp=bdp)$root
        b0<-bfct_mvexp(c0,p,df)
        
        # tau constant
        tmp_alpha<- 1/p*(Sfct_mvexp(c0,e=2,p,df) -4/c0^2*Sfct_mvexp(c0,e=4,p,df) + 6/c0^4*Sfct_mvexp(c0,e=6,p,df) -4/c0^6*Sfct_mvexp(c0,e=8,p,df) + 1/c0^8*Sfct_mvexp(c0,e=10,p,df))
        tmp_beta<-  -2/c0^2*(1 + 2/p)*Sfct_mvexp(c0,e=2,p,df) + 1/c0^4*(1 + 4/p)*Sfct_mvexp(c0,e=4,p,df)+Sfct_mvexp(c0,e=0,p,df)
        tau= tmp_alpha/tmp_beta^2
        
        # sigma1 constant
        g1<-Sfct_mvexp(c0,e=2,p,df)*(p+2)-2/c0^2*(p+4)*Sfct_mvexp(c0,e=4,p,df) +1/c0^4*(p+6)*Sfct_mvexp(c0,e=6,p,df)
        sigma1<- p*(p+2)*1/g1^2*(Sfct_mvexp(c0,e=4,p,df)-4/c0^2*Sfct_mvexp(c0,e=6,p,df)+6/c0^4*Sfct_mvexp(c0,e=8,p,df)-4/c0^6*Sfct_mvexp(c0,e=10,p,df) +1/c0^8*Sfct_mvexp(c0,e=12,p,df))
        
        # sigma2 constant 
        tmp_num<-0.25*Sfct_mvexp(c0,e=4,p,df) -1/(2*c0^2)*Sfct_mvexp(c0,e=6,p,df) +5/(12*c0^4)*Sfct_mvexp(c0,e=8,p,df)-1/(6*c0^6)*Sfct_mvexp(c0,e=10,p,df) +1/(36*c0^8)*Sfct_mvexp(c0,e=12,p,df)+c0^4/36*(1-Sfct_mvexp(c0,e=0,p,df))-b0^2
        g3<-Sfct_mvexp(c0,e=2,p,df)-2/c0^2*Sfct_mvexp(c0,e=4,p,df) +1/c0^4*Sfct_mvexp(c0,e=6,p,df)
        sigma2= -2/p*sigma1 +4*tmp_num/g3^2
        
        Cte<-2*sigma1+sigma2
        
      }else{stop("ASV for this estimator is not available")}
    }
    ASV<- tau*g^4 + g^2/4*Cte
  }
  return(ASV)
}



#------------------- Auxiliary functions
DistMahala<-function(x,mu,sigma)
{
  return((x-mu)%*%solve(sigma) %*% (x-mu))
}



myIFrewMCDLoc_matbis<-function(x,mu,sigma,mod,alpha,d1,d2,qdelta,df){
  ech<-lapply(seq_len(nrow(x)), function(i) x[i,]) # transform the matrix in a list of rows
  p<-length(x)
  
  l1<- lapply(lapply(ech,myIFMCDloc,mu=mu,sigma=sigma,mod=mod,alpha=alpha,df=df),function(y) d2/d1*y)
  l2<- lapply(ech,function(x) (as.numeric(t(x-mu)%*% solve(sigma)%*%(x-mu))<qdelta)/d1*(x-mu))
  ifT<-lapply(seq_len(nrow(x)),function(i)l1[[i]]+l2[[i]])
  return(ifT)
}

myIFMCDloc<-function(x,mu,sigma,mod,alpha,df=NULL){
  # INPUT:
  # x : vector in which to compute the IF
  # mu: mean vector
  # sigma: covariance matrix
  # mod: distribution "norm", "stud" or "exp"
  # alpha: BDP 
  
  p<-length(x)
  if (mod[1]=="norm")
  {
    qalpha<-qchisq(1-alpha,p)
    C1<- 1/pchisq(qalpha,p+2)
    
  }else if (mod[1]=="stud"){
    K<- (df-2)^(-p/2)*gamma((p+df)/2)/(pi^(p*0.5)*gamma(df/2))
    const<- (1-alpha)*gamma(0.5*p)/(2*pi^(p*0.5))*1/K
    qalpha<-uniroot(stud_int2,interval=c(0,30),p=p,df=df,const=const)$root
    C1<-(- 2/const*(1-alpha)/p *(-0.5*(df+p))/(df-2)*stud_funct2(qalpha,e=2,p=p,df=df))^(-1)
    
  }else if(mod[1]=="powerexp"){
    const<-(1-alpha)*gamma(p/(2*df))/(2*df) *((p*gamma(p/(2*df)))/gamma((p+2)/(2*df)))^(0.5*p)
    a=(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
    qalpha<-uniroot(mvexp_int2,interval=c(1,30),p=p,df=df,const=const)$root
    C1<- (-2/const*(1-alpha)/p*(-a*df)*mvexp_funct(qalpha,e=2*df,p=p,df=df))^(-1)
    
  }  else return("Not an appropriate model")
  indic<- ((t(x-mu) %*% solve(sigma) %*%(x-mu) )< qalpha)
  return((x-mu)* C1*indic)
}


myIFMCDCov<-function(x,mu,sigma,mod,alpha,df=NULL){
  # INPUT:
  # x : vector in which to compute the IF
  # mu: mean vector
  # sigma: covariance matrix
  # mod: distribution "norm", "stud" or "exp"
  # df: degree of freedom fr "stud" or "exp" distributions
  # alpha: BDP 
  
  p<-length(x)
  if (mod[1]=="norm"){
    qalpha<-qchisq(1-alpha,p)
    calpha<-(1-alpha)/pchisq(qalpha,p+2)
    c2<- -pchisq(qalpha,p+2)/2
    c3<- -pchisq(qalpha,p+4)/2
    
  }else if (mod[1]=="stud"){
    K<- (df-2)^(-p/2)*gamma((p+df)/2)/(pi^(p*0.5)*gamma(df/2))
    const<- (1-alpha)*gamma(0.5*p)/(2*pi^(p*0.5))*1/K
    qalpha<-uniroot(stud_int2,interval=c(0,30),p=p,df=df,const=const)$root
    calpha<- (1/const *1/p*stud_funct(qalpha,e=2,p=p,df=df))^-1
    c3<-1/const*(1-alpha)/(p*(p+2))*(-0.5*(df+p))/(df-2) * stud_funct2(qalpha,e=4,p=p,df=df)
    c2<-1/const*(1-alpha)/p*(-0.5*(df+p))/(df-2)*stud_funct2(qalpha,e=2,p=p,df=df)
    
  }else if (mod[1]=="powerexp"){
    const<-(1-alpha)*gamma(p/(2*df))/(2*df) *((p*gamma(p/(2*df)))/gamma((p+2)/(2*df)))^(0.5*p)
    a=(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
    qalpha<-uniroot(mvexp_int2,interval=c(1,30),p=p,df=df,const=const)$root
    calpha<-(1/const*1/p*mvexp_funct(qalpha,e=2,p,df))^-1
    c3<-1/const*(1-alpha)/(p*(p+2))*(-a*df) *mvexp_funct(qalpha, e=2*df+2,p=p,df=df)
    c2<-1/const*(1-alpha)/p *(-a*df)*mvexp_funct(qalpha,e=2*df,p=p,df=df)
    
  }else return("Not an appropriate model")
  
  b1<- -2*calpha*c3/(1-alpha)
  b2<- 0.5 + calpha/(1-alpha)*(c3- qalpha/p*(c2+(1-alpha)/2) )
  
  C2<- -1/(2*c3)
  C3<-  b2/(b1-p*b2) * 1/(2*c3)
  C4<-  1/(b1-p*b2)*(calpha*qalpha)/((1-alpha)*p)
  C5<- - 1/(b1-p*b2)*(calpha*qalpha/p -1 )
  
  distx<- c(t(x-mu)%*% solve(sigma) %*% (x-mu))
  indic<-as.integer(distx < qalpha)
  funalpha<-C2*indic
  funbeta<- C3*indic*distx + C4*indic + C5
  ifC<-funalpha* (x-mu) %*% t(x-mu) - funbeta *sigma
  return(ifC)
}

myIFrewMCDCov_mat<-function(x,mu,sigma,mod,alpha, delta,df=NULL){
  # INPUT:
  # x :matrix on which you want to comupte the IF for each row
  # mu: mean vector
  # sigma: covariance matrix
  # mod: distribution "norm", "stud" or "exp"
  # df: degree of freedom fr "stud" or "exp" distributions
  # alpha: BDP 
  # delta: parameter for the weight function
  
  ech<-lapply(seq_len(nrow(x)), function(i) x[i,]) # transform the matrix in a list of rows
  p<-dim(x)[2]
  if (mod[1]=="norm"){
    qdelta<-qchisq(1-delta,p)
    d3<- pchisq(qdelta,p+2)
    d4<- d3 - pchisq(qdelta,p+4)
    
  }else if (mod[1]=="stud"){
    K<- (df-2)^(-p/2)*gamma((p+df)/2)/(pi^(p*0.5)*gamma(df/2))
    const<- (1-delta)*gamma(0.5*p)/(2*pi^(p*0.5))*1/K
    qdelta<-uniroot(stud_int2,interval=c(0,30),p=p,df=df,const=const)$root
    d3<- 1/const*(1-delta)/p*stud_funct(qdelta,e=2,p=p,df=df)
    d4<- d3+ 1/const*(1-delta)*2/(p*(p+2))*(-0.5*(df+p))/(df-2) * stud_funct2(qdelta,e=4,p=p,df=df)
    
  }else if (mod[1]=="powerexp"){
    const<-(1-delta)*gamma(p/(2*df))/(2*df) *((p*gamma(p/(2*df)))/gamma((p+2)/(2*df)))^(0.5*p)
    a=(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
    qdelta<-uniroot(mvexp_int2,interval=c(1,30),p=p,df=df,const=const)$root
    d3<-1/const*(1-delta)/p*mvexp_funct(qdelta, e=2,p=p,df=df)
    d4<-d3+ 1/const*(1-delta)*2/(p*(p+2)) *(-a*df)*mvexp_funct(qdelta,e=2*df+2,p=p,df=df)    
  }
  else return("Not an appropriate model")
  
  l1<-lapply(lapply(ech, myIFMCDCov,mu=mu,sigma=sigma,mod=mod,alpha=alpha,df=df),function(y)d4/d3*y + d4/(2*d3)*sum(diag(solve(sigma)%*%y))* sigma)
  l2<- lapply(ech, function(x) (as.numeric(t(x-mu)%*% solve(sigma)%*%(x-mu))<qdelta)/d3*(x-mu)%*%t(x-mu)-sigma)
  ifC<-lapply(seq_len(nrow(x)),function(i)l1[[i]]+l2[[i]])
  return(ifC)
}


#----------------- Generate samples form the multiv power exponential
# because the generation already available was wrong

myrmvpowerexp<-function (n, Location = rep(0, nrow(Scatter)), Scatter = diag(length(Location)), Beta = 1) {
  p <- length(Location)
  if (!isSymmetric(Scatter, tol = sqrt(.Machine$double.eps))) {
    stop("Scatter must be a symmetric matrix")
  }
  if (p != nrow(Scatter)) {
    stop("Location and Scatter have non-conforming size")
  }
  ev <- eigen(Scatter, symmetric = TRUE)
  if (!all(ev$values >= -sqrt(.Machine$double.eps) * abs(ev$values[1]))) {
    warning("Scatter is numerically not positive definite")
  }
  ScatterSqrt <- ev$vectors %*% diag(sqrt(ev$values), length(ev$values)) %*% t(ev$vectors)
  radius <- (rgamma(n, shape = p/(2 * Beta), scale = 2))^(1/(2 *  Beta))
  un <- runifsphere(n = n, p = p)
  mvpowerexp <- radius * un %*% ScatterSqrt
  mvpowerexp <- sweep(mvpowerexp, 2, Location, "+")
  return(mvpowerexp)
}


